// Block mode radio button handling
document.addEventListener('DOMContentLoaded', () => {
  const name = 'blockMode';

  chrome.storage.sync.get([name], (result) => {
    const savedValue = result[name];

    const radios = document.getElementsByName(name);
    radios.forEach((radio) => {
      if (savedValue && radio.value === savedValue) {
        radio.checked = true;
      }

      radio.addEventListener('change', () => {
        if (radio.checked) {
          chrome.storage.sync.set({ [name]: radio.value });
        }
      });
    });
  });
});


// Font selection handling
const fontSelect = document.querySelector('select[name="fontselect"]');

function isLocalFontAvailable(fontName) {
    if (fontName === 'default') return true;
    const canvas = document.createElement("canvas");
    const context = canvas.getContext("2d");
    const text = "abcdefghijklmnopqrstuvwxyz0123456789";
    const testSize = "72px";
    const realName = fontName.replace(/\+/g, ' ');

    const getWidth = (family) => {
        context.font = `${testSize} ${family}`;
        return context.measureText(text).width;
    };

    const monoWidth = getWidth("monospace");
    const sansWidth = getWidth("sans-serif");
    const serifWidth = getWidth("serif");

    context.font = `${testSize} "${realName}", monospace`;
    const tMono = context.measureText(text).width;
    context.font = `${testSize} "${realName}", sans-serif`;
    const tSans = context.measureText(text).width;
    context.font = `${testSize} "${realName}", serif`;
    const tSerif = context.measureText(text).width;

    return tMono !== monoWidth || tSans !== sansWidth || tSerif !== serifWidth;
}

async function syncAndRestore() {
    const res = await chrome.storage.local.get(['selectedFont']);
    const saved = res.selectedFont || 'default';

    const processOptions = () => {
        const options = Array.from(fontSelect.options);
        if (options.length <= 1) return false;

        options.forEach(opt => {
            if (opt.value !== 'default' && !isLocalFontAvailable(opt.value)) {
                opt.remove();
            }
        });

        const exists = Array.from(fontSelect.options).some(o => o.value === saved);
        fontSelect.value = exists ? saved : 'default';
        return true;
    };

    const observer = new MutationObserver(() => {
        if (processOptions()) observer.disconnect();
    });

    observer.observe(fontSelect, { childList: true });

    setTimeout(() => {
        processOptions();
        observer.disconnect();
    }, 2000);
}

function applyFontToTab(val) {
    chrome.tabs.query({ active: true, currentWindow: true }, (tabs) => {
        const tab = tabs[0];
        if (!tab?.id || !tab.url || !tab.url.includes("x.com")) return;

        const isDef = val === 'default';
        const name = isDef ? '' : val.replace(/\+/g, ' ');

        chrome.scripting.executeScript({
            target: { tabId: tab.id },
            func: (fName, isRemove) => {
                const id = 'custom-font-style';
                const old = document.getElementById(id);
                if (isRemove) {
                    if (old) old.remove();
                    return;
                }
                let style = old || document.createElement('style');
                style.id = id;
                style.textContent = `* , body , * * , .r-1tl8opc { font-family: "${fName}" !important; }`;
                if (!old) document.head.appendChild(style);
            },
            args: [name, isDef]
        });
    });
}

fontSelect.addEventListener('change', (e) => {
    const val = e.target.value;
    chrome.storage.local.set({ selectedFont: val }, () => {
        applyFontToTab(val);
    });
});

syncAndRestore();